from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector, \
    get_var_assignment_matches_containing_url_alias, \
    contains_alias
from codeable_detectors.evidences import FailedEvidence, LinkEvidence
from codeable_detectors.java.java_detectors import get_java_variable_assignments
from codeable_detectors.utils import get_required_keyword_arg
from codeable_detectors.pyparsing_patterns import round_braces_block


class JavaApacheHTTPLink(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["java"]
        self.new_http_method_pattern = Literal("new") + (Literal("HttpPost") | Literal("HttpGet") |
                                                         Literal("HttpPut") | Literal(
                    "HttpDelete")) + round_braces_block + Literal(";")

    def detect_in_context(self, ctx, **kwargs):
        target = get_required_keyword_arg('target', kwargs)
        matches = []

        new_http_method_matches = ctx.matches_pattern(self.new_http_method_pattern)
        if not new_http_method_matches:
            return FailedEvidence("apache http method creation not found")

        for new_http_method_match in new_http_method_matches:
            parameters = new_http_method_match.text[
                         new_http_method_match.text.find("(") + 1:new_http_method_match.text.find(")")]
            if contains_alias(parameters, target.aliases):
                matches.append(new_http_method_match)
            else:
                # maybe a variable in the HTTP method contains the url to the alias
                var_assignments_matching_aliases = get_var_assignment_matches_containing_url_alias(
                    get_java_variable_assignments(ctx),
                    target.aliases)
                for var, varAssignmentMatch in var_assignments_matching_aliases:
                    if var in parameters:
                        matches.append(new_http_method_match)
                        matches.append(varAssignmentMatch)

        if matches:
            return LinkEvidence(matches).set_properties(detector_link_types="restfulHTTP",
                                                        detector_technology_types=["java", "apacheHTTPClient"],
                                                        kwargs=kwargs)
        return FailedEvidence("no apache http method call to target found")
